# SIMULATED ANNEALING FOLDING PROTOCOL 01/2015
#===============================================================================
# to run type: xplor -py -o fold.out fold_REPEL.py
#			   xplor -py -smp 10 -o fold.out fold_REPEL.py
#===============================================================================

xplor.requireVersion('2.40')

import protocol

# TOTAL NUMBER OF STRUCTURES AND BASE NAME FOR OUTPUT PDB FILES.
Nstructures = 100
outFilename = "SCRIPT_STRUCTURE.sa"

# INITIALIZE RANDOM SEED.
protocol.initRandomSeed(3421)

# INITIALIZE PARAMETERS AND TOPOLOGY.

# READ MODEL, GENERATE PSF AND GENERATE EXTENDED STRUCTURE.
import psfGen
psfGen.pdbToPSF("Ail_template.pdb")
protocol.genExtendedStructure()

# SET UP PARAMETERS, POTLISTS AND RAMPS FOR STRUCTURE CALCULATIONS.
#===============================================================================
pots = PotList()

from simulationTools import StaticRamp, MultRamp
from simulationTools import InitialParams
highTempParams1  = []                   # Settings for high T MD stage.
highTempParams2  = []                   # Settings for high T MD stage.
highTempParams3 = []                   # Settings for high T MD stage.
rampedParams = []                   # Settings for annealing stage.



# SET UP ALIGNMENT TENSOR AND TENSOR CALCULATION DURING SIMULATED ANNEALING.
# rdcPotTools uses NH bond distance r(NH)=1.042A (Da=10.76; DaMAX = 21523.28).
# To calculate tensor orientation, Da and Rh:   calcTensor(oTensor).
# To calculate tensor orientation only:         calcTensorOrientation(oTensor).
# To analyze pot terms with given setFreedom:   VarTensor_analyze(pots).
# To fix all tensor parameters use:             oTensor.setFreedom("fix").
#===============================================================================
memfreedom="fix"
oT1freedom="fix"

tensorInit = []                    # Settings for alignment tensor.
tensorFinal = []                    # Final settings for alignment tensor.
if False:
    from varTensorTools import create_VarTensor
    from varTensorTools import calcTensorOrientation
    from simulationTools import analyze
    mem = create_VarTensor("mem")   # This is the main membrane order tensor
    mem.setDa(10.76)                # based on rNH=1.042.
    mem.setRh(0.00)                 # axially symmetric bilayer.
    
    oT1 = create_VarTensor("oT1")   # For extra-membrane parts with potentially different Da
    oT1.setDa(10.76)                # based on rNH=1.042.
    oT1.setRh(0.00)                 # axially symmetric bilayer.
    
    tensorInit.append(StaticRamp("calcTensorOrientation(mem)"))
    from varTensorTools import alignTensorToZ
    tensorInit.append(StaticRamp("alignTensorToZ(mem)"))
    tensorInit.append(StaticRamp("print analyze(pots)"))
    tensorInit.append(StaticRamp("mem.setFreedom(memfreedom)"))
    
    tensorInit.append(StaticRamp("copyTensor(oT1,mem)"))
    tensorInit.append(StaticRamp("oT1.setFreedom(oT1freedom)"))
    tensorInit.append(StaticRamp("print analyze(pots)"))
    
    rampedParams.append(StaticRamp("print analyze(pots)"))
    tensorFinal.append(StaticRamp("print analyze(pots)"))

# EXPERIMENTAL AND STATISTICAL RESTRAINING TERMS.
#===============================================================================
if False:
    rdcs = PotList('rdc')
    from rdcPotTools import create_RDCPot
    for (name, file, tensor, scale) in [('rdcNHtm', rdcNH1_data, mem, 1),
                                        ('rdcNHem', rdcNH2_data, mem, 1)]:
        rdc=create_RDCPot(name=name, file=file, oTensor=tensor)
        rdc.setScale(scale)
        rdc.setShowAllRestraints(1)
        rdcs.append(rdc)
    pots.append(rdcs)
    rampedParams.append(MultRamp(0.01, 1, "rdcs.setScale(VALUE)"))

if False:
    csas = PotList('csa')
    from csaPotTools import create_CSAPot
    for (name, file, tensor, scale) in [('csaNtm',  csaN1_data,  mem,   1),
                                        ('csaNGtm', csaN1g_data, mem,   1),
                                        ('csaNem',  csaN2_data,  oT1, 1),
                                        ('csaNGem', csaN2g_data, oT1, 1)]:
        csa=create_CSAPot(name=name, file=file, oTensor=tensor)
        csa.setScale(scale)
        csa.setShowAllRestraints(1)
        csa.setVerbose(True)
        csa.setDaScale(-21523.28)       # based on rNH=1.042 ***NOTE SIGN.
        csas.append(csa)
    pots.append(csas)
    rampedParams.append(MultRamp(0.01, 0.1, "csas.setScale(VALUE)"))

if True:
    dsts = PotList('dst')
    from noePotTools import create_NOEPot
    for (exp, file, scale) in [('noe', "noe_Ail.tbl", 1),
                               ('hbn', "hbn_Ail.tbl", 1)]:
        dst = create_NOEPot(exp,file,)
        dst.setScale(scale)
        dsts.append(dst)
    pots.append(dsts)
    rampedParams.append(MultRamp(2, 30, "dsts.setScale(VALUE)"))

from xplorPot import XplorPot

# Dihedral restraint term
#
cdi_data="cdi_Ail.tbl"
protocol.initDihedrals(cdi_data)
from xplorPot import XplorPot
pots.append(XplorPot('CDIH'))
highTempParams1.append(StaticRamp("pots['CDIH'].setScale(100)"))
highTempParams2.append(StaticRamp("pots['CDIH'].setScale(100)"))
highTempParams3.append(StaticRamp("pots['CDIH'].setScale(100)"))
rampedParams.append(StaticRamp("pots['CDIH'].setScale(200)"))

# CONFORMATIONAL ENERGY TERMS.
#===============================================================================
from torsionDBPotTools import create_TorsionDBPot
torsionDB = create_TorsionDBPot(name='torsionDB')
pots.append(torsionDB)
rampedParams.append(MultRamp(0.002, 2, "torsionDB.setScale(VALUE)"))

from xplorPot import XplorPot
pots.append(XplorPot('BOND'))          # Dflt scale [1]
pots.append(XplorPot('ANGL'))          # Dflt scale [1]
pots.append(XplorPot('IMPR'))          # Dflt scale [1]
rampedParams.append(MultRamp(1  , 1,  "pots['BOND'].setScale(VALUE)"))
rampedParams.append(MultRamp(0.4, 1,  "pots['ANGL'].setScale(VALUE)"))
rampedParams.append(MultRamp(0.1, 1,  "pots['IMPR'].setScale(VALUE)"))

# THRESHOLD VALUES FOR  VIOLATION ANALYSIS OF THE POTLIST TERMS.
# Use default values unless specified (bond [0.05], cdih [5.0], noe [0.5], rdc [0.5]).
#===============================================================================
pots['BOND'].setThreshold(0.05)     # dflt [0.05]
pots['ANGL'].setThreshold(5.00)     # dflt [2.0]
pots['IMPR'].setThreshold(5.00)     # dflt [2.0]
pots['CDIH'].setThreshold(5.00)     # dflt [2.0]


# NONBONDED ENERGY TERM - REPEL.
# Parameters differ between the high temperature and annealing stages.
# For standard repel atomic overlap is not fatal. Therefore, we allow
# most atoms to move through each other (onlyCA=1) in the initial stages
# of the calculation and then turn on all-atom interactions (onlyCA=0)
# during simulated annealing.
#
# Dflt settings:
# [cutnb=4.5, rcon=4.0, nbxmod=3, selStr='all', tol=0.5, repel=0.8, onlyCA=0, simulation=0]
# rcon = energy constant for REPEL function.
# repel = factor by which to multiply the vdw radius.
#
# nbxmod=5: Use with no torsionDB. Exclude 1-2, 1-3 and compute 1-4 interactions.
# nbxmod=4: Use with torsionDB. Exclude 1-2, 1-3 and 1-4 interactions.
#===============================================================================
if True:
    pots.append(XplorPot('VDW'))          # dflt scale [1]

    highTempParams1.append(StaticRamp("pots['VDW'].setScale(0.004)"))
    highTempParams1.append(StaticRamp("""protocol.initNBond(cutnb=100,
                                                         repel=1.2,
                                                         nbxmod=4,
                                                         tolerance=45,
                                                         onlyCA=1)"""))
    # standard all-atom repel
    rampedParams.append(StaticRamp("protocol.initNBond(nbxmod=4)"))
    rampedParams.append( MultRamp(0.9,0.8,
                       "xplor.command('param nbonds repel VALUE end end')") )
    rampedParams.append( MultRamp(.004,4,
                                  "pots['VDW'].setScale(0)") )

# NONBONDED ENERGY TERM - EEFx/IMMx IMPLICIT SOLVATION.
# Membrane hydrophobic thickness: DMPC (25.4A), DPPC (28.6A), POPC (27.0A), DOPC (29.6A)
# See Marsh, Handbook of lipid bilayers, 2nd Ed. p. 379.
# IMM_n<10 generates a steeper polar-apolar membrane transition gradient.
# IMM_a>0.85 increases dielectric screening (reduces electrostatic effects) in membrane.
# EEFx/IMMx uses nbxmod=4 (exclude 1-2, 1-3 and 1-4 interactions).
#===============================================================================
#from eefxPotTools import create_EEFxPot, param_LK
#eefx=create_EEFxPot("eefx","ALL",paramSet=param_LK)
#eefx.setScale(1)
##eefx.setVerbose(1)
#eefx.setIMMx(True)
#eefx.useGROUp(1)
#eefx.setMoveTol(0.5)
#print eefx.showParam()
#eefx.setThickness(25) # IMMx membrane thickness [25.4 DMPC; 28.6 DPPC; 27.0 POPC; 29.6 DOPC].
#eefx.setProfileN(10)  # IMMx n parameter of membrane profile (use n<3 in early stages).
#eefx.setA(0.85)       # IMMx a value that scales dielectric screening. Dflt=[0.85].
#pots.append(eefx)
#
#highTempParams1.append(StaticRamp("eefx.setScale(0)"))
#highTempParams3.append(StaticRamp("eefx.setScale(0.004)"))
#rampedParams.append(MultRamp(0.004,1.0,"eefx.setScale(VALUE)"))

from eefxPotTools import setCenter, setCenterXY
IMM_com = "resid 1:156 and (name CA)"   # Center of mass selection for IMMx position.
Zpos=0									# Z position relative to IMMx membrane center. Dflt=[0].
setCenter(IMM_com, Zpos)				# Translate selected center of mass to IMMx Zpos.


# SET UP IVM OBJECTS (dyn, minc) THAT PERFORM DYNAMICS AND MINIMIZATION.
# IVM (internal variable module) is used to perform dynamics and minimization in
# both torsion-angle and Cartesian space. Bonds, angles and many impropers cannot
# change with internal torsion-angle dynamics.
#===============================================================================
from ivm import IVM
dyn = IVM()                     # IVM object for torsion-angle dynamics.
dyn.reset()                     # reset ivm topology for torsion-angle dynamics.
protocol.torsionTopology(dyn)

minc = IVM()                    # IVM object for Cartesian minimization.
protocol.cartesianTopology(minc)


# Calculate initial tensor parameters.
#===============================================================================
InitialParams(tensorInit)      # Initial tensor parameters.


# DYNAMICS SETTINGS FOR HIGH T AND ANNEALING STAGES.
#===============================================================================
ini_temp = 3500.0   ; fin_temp = 25.0   # Initial and final temperatures.
protocol.massSetup()                    # Give atoms uniform weights except for axes.


# CALCULATE STRUCTURE MODULE.
#===============================================================================
def calcOneStructure(loopInfo):
    """
    This function calculates a single structure, performs analysis on the
    structure and then writes out a pdb file with remarks.
    """
    # Generate initial structure and minimize.
    #===========================================================================
    # Generate initial structure by randomizing torsion angles.
    # Then set torsion angles from restraints (this shortens high T dynamics).
    # Then set selected center of mass to membrane center (IMMx z=0).
    import monteCarlo
    monteCarlo.randomizeTorsions(dyn)
    protocol.fixupCovalentGeom(maxIters=100, useVDW=1)
    import torsionTools
    torsionTools.setTorsionsFromTable(cdi_data)
   
    InitialParams(rampedParams)     # parameters for SA.
    InitialParams(highTempParams1)  # reset some rampedParams.
    setCenter(IMM_com, Zpos)        # Translate selected center of mass to IMMx Zpos.

    # Torsion angle minimization.
    protocol.initMinimize(dyn,
                          potList=pots,
                          numSteps=100,
                          printInterval=50)
    dyn.run()

    # High temperature dynamics.
    #===========================================================================
    # High temperature dynamics stage 1.
    protocol.initDynamics(dyn,
                          potList=pots,         # potential terms to use.
                          bathTemp=ini_temp,    # set bath temperature.
                          initVelocities=1,     # uniform initial velocities.
                          finalTime=3,          # run for finalTime or
                          numSteps=3001,        # numSteps * 0.001, whichever is less.
                          printInterval=100)    # printing rate in steps.
    dyn.setETolerance(ini_temp/100)             # used to det. stepsize, dflt [temp/1000].
    dyn.run()

    # High temperature dynamics stage 2.
    InitialParams(highTempParams2)    
    setCenter(IMM_com, Zpos)            # translate selected center of mass to IMMx Zpos.
    protocol.initDynamics(dyn,
                          potList=pots,         # potential terms to use.
                          bathTemp=ini_temp,    # set bath temperature.
                          initVelocities=1,     # uniform initial velocities.
                          finalTime=3,          # run for finalTime or
                          numSteps=3001,        # numSteps * 0.001, whichever is less.
                          printInterval=100)    # printing rate in steps.
    dyn.setETolerance(ini_temp/100)             # used to det. stepsize, dflt [temp/1000].
    dyn.run()
    
    # High temperature dynamics stage 3.
    InitialParams(highTempParams3)
    setCenter(IMM_com, Zpos)            # translate selected center of mass to IMMx Zpos.
    protocol.initDynamics(dyn,
                          potList=pots,         # potential terms to use.
                          bathTemp=ini_temp,    # set bath temperature.
                          initVelocities=1,     # uniform initial velocities.
                          finalTime=26,			# run for finalTime or
                          numSteps=26001,		# numSteps * 0.001, whichever is less.
                          printInterval=100)    # printing rate in steps.
    dyn.setETolerance(ini_temp/100)             # used to det. stepsize, dflt [temp/1000].
    dyn.run()
                          
    # Initialize integrator and loop for simulated annealing and run.
    #===========================================================================
    # Dynamics for annealing.
    setCenter(IMM_com, Zpos)            # translate selected center of mass to IMMx Zpos.
    protocol.initDynamics(dyn,
                          potList=pots,
                          finalTime=0.4,        # run for finalTime or
                          numSteps=201,         # numSteps*0.001, whichever is less.
                          printInterval=100)    

    # Set up cooling loop and run.
    from simulationTools import AnnealIVM
    AnnealIVM(initTemp=ini_temp,
                       finalTemp=fin_temp,
                       tempStep=12.5,
                       ivm=dyn,
                       rampedParams=rampedParams
                       ).run()

    # Final minimization.
    #===========================================================================
    # Torsion angle minimization.
    protocol.initMinimize(dyn,
                          numSteps=500,         # dflt [500 steps]
                          potList=pots,
                          printInterval=50)
    dyn.run()

    # Final Cartesian minimization.
    protocol.initMinimize(minc,
                          numSteps=500,         # dflt [500 steps]
                          potList=pots,
                          dEPred=10)
    minc.run()

    # Recenter coordinates in XY plane.
    InitialParams(tensorFinal)
    setCenter(IMM_com, Zpos)           # translate selected center of mass to IMMx  Zpos.
    #setCenterXY()                       # translate protein coordinates to XY center.
    
    # Do analysis and write structure when this routine is finished.
    pass

# LOOP CONTROL.
#===========================================================================
from simulationTools import StructureLoop, FinalParams
StructureLoop(structLoopAction=calcOneStructure,
              numStructures=Nstructures,
              pdbTemplate=outFilename,
              doWriteStructures=1,              # analyze and write coords after calc
              genViolationStats=1,              # print stats file
              averageContext=FinalParams(rampedParams),
              averageFitSel="",                 # selection for bkbn rmsd [CA]
              averageCompSel="",                # selection for heavy atom rmsd
              averageTopFraction=0.1,           # Report stats on top 10%
              averagePotList=pots,              # Terms for stats and avg
              averageSortPots=[                 # Terms used to sort models
                               term for term in pots \
                               if term.instanceName() not in ("torsionDB",
                                                               "VDW"      )],
              ).run()
