# SIMULATED ANNEALING FOR REFINEMENT PROTOCOL WITH EEFx 01/2015
#===============================================================================
# to run type: xplor -py -o refine.out refine_EEF.py
#			   xplor -py -smp 10 -o refine.out refine_EEF.py
#===============================================================================

xplor.requireVersion('2.40')

# TOTAL NUMBER OF STRUCTURES AND BASE NAME FOR OUTPUT PDB FILES.
Nstructures = 100
outFilename = "SCRIPT_STRUCTURE.pdb"

orderedRegion="""resid 3:15 or resid 27:36 or resid 41:51 or resid 68:80 or
                 resid 85:98 or resid 110:126 or resid 129:139 or resid 144:155"""

# INITIALIZE RANDOM SEED.
import protocol
protocol.initRandomSeed(3421)

useEEFx=True # Set to 1 for EEFx/IMMx , false for the REPEL potential

# initialize parameters, topology and starting model.
if useEEFx:   
    protocol.parameters['protein']="eefx/protein_eef.par"
    protocol.topology['protein']  ="eefx/protein_eef22.top"
    pass

inputStructures="refine_EEF_*.sa"

from glob import glob
ini_model=glob(inputStructures)[0] #structure used in initializing energy terms

protocol.loadPDB(ini_model, deleteUnknownAtoms=True)

#this next line may significantly change the structure
#protocol.fixupCovalentGeom(maxIters=100,useVDW=1)


# SET UP POTLISTS, PARAMETERS AND FORCE CONSTANTS TO RAMP DURING STRUCTURE CALCULATIONS.
#===============================================================================
pots = PotList()

from simulationTools import StaticRamp, MultRamp
from simulationTools import InitialParams
highTempParams = []                 # Settings for high T MD stage.
rampedParams = []                   # Settings for annealing stage.
tensorInit = []                     # Settings for alignment tensor.
tensorFinal = []                    # Final settings for alignment tensor.

# compare atomic Cartesian rmsd with a reference structure
#  backbone and heavy atom RMSDs will be printed in the output
#  structure files
#
alignedResids = """resid 29:41 or resid 53:62 or resid 67:77 or resid 94:106
                   or resid 111:124 or resid 136:152 or resid 155:165 or
                   resid 170:181"""
from posDiffPotTools import create_PosDiffPot
refRMSD = create_PosDiffPot("refRMSD",
                            "(%s) and (name CA or name C or name N)"%orderedRegion,
                            "(%s) and (name CA or name C or name N)"%alignedResids,
                            pdbFile='3qra.pdb',
                            cmpSel=None)



# SET UP ALIGNMENT TENSOR AND TENSOR CALCULATION DURING SIMULATED ANNEALING.
# rdcPotTools uses NH bond distance r(NH)=1.042A (Da=10.76; DaMAX = 21523.28).
# To calculate tensor orientation, Da and Rh:   calcTensor(oTensor).
# To calculate tensor orientation only:         calcTensorOrientation(oTensor).
# To analyze pot terms with given setFreedom:   VarTensor_analyze(pots).
# To fix all tensor parameters use:             oTensor.setFreedom("fix").
#===============================================================================
memfreedom="fixAxis, varyDa, fixRh"

if True:
    from varTensorTools import create_VarTensor, alignTensorToZ
    from varTensorTools import calcTensorOrientation
    from simulationTools import analyze
    
    mem = create_VarTensor("mem")   # This is the main membrane order tensor
    mem.setDa(10.76)                # based on rNH=1.042.
    mem.setRh(0.00)                 # axially symmetric bilayer.
    
    tensorInit.append(StaticRamp("calcTensorOrientation(mem)"))
    tensorInit.append(StaticRamp("alignTensorToZ(mem)"))
    tensorInit.append(StaticRamp("print analyze(pots)"))
    tensorInit.append(StaticRamp("mem.setFreedom(memfreedom)"))
    rampedParams.append(StaticRamp("print analyze(pots)"))
    tensorFinal.append(StaticRamp("print analyze(pots)"))



# EXPERIMENTAL AND STATISTICAL RESTRAINING TERMS.
#===============================================================================
if False:
    from rdcPotTools import create_RDCPot
    rdcs = PotList('rdc')
    for (name, file, tensor, scale) in [('rdcNH_F', rdcNH1_data, mem, 1)]:
        rdc=create_RDCPot(name=name, file=file, oTensor=tensor)
        rdc.setScale(scale)
        rdc.setThreshold(1.0)     # dflt [0.0]
        rdc.setShowAllRestraints(1)
        rdcs.append(rdc)
    pots.append(rdcs)
    rampedParams.append(MultRamp(0, 0, "rdcs.setScale(VALUE)"))

if False:
    csas = PotList('csa')
    for (name, file, tensor, scale) in [('csaN_F', csaN1_data,  mem, 1)]:
        csa=create_CSAPot(name=name, file=file, oTensor=tensor)
        csa.setScale(scale)
        rdc.setThreshold(1.0)     # dflt [0.0]
        csa.setShowAllRestraints(1)
        csa.setVerbose(True)
        csa.setDaScale(-21523.28)       # based on rNH=1.042 ***NOTE SIGN.
        csas.append(csa)
    pots.append(csas)
    rampedParams.append(MultRamp(0, 0, "csas.setScale(VALUE)"))

if False:
    rdcsCross = PotList('rdcCross')
    for (name, file, tensor, scale) in [('rdcNH_V', rdcNH2_data, mem, 1)]:
        rdcCross=create_RDCPot(name=name, file=file, oTensor=tensor)
        rdcCross.setScale(scale)
        rdcCross.setShowAllRestraints(1)
        rdcsCross.append(rdcCross)
    pots.append(rdcsCross)
    rampedParams.append(MultRamp(0, 0, "rdcsCross.setScale(VALUE)"))

if False:
    from csaPotTools import create_CSAPot
    csasCross = PotList('csaCross')
    for (name, file, tensor, scale) in [('csaN_V', csaN2_data,  mem, 1)]:
        csaCross=create_CSAPot(name=name, file=file, oTensor=tensor)
        csaCross.setScale(scale)
        csaCross.setShowAllRestraints(1)
        csaCross.setVerbose(True)
        csaCross.setDaScale(-21523.28)       # based on rNH=1.042 ***NOTE SIGN.
        csasCross.append(csaCross)
    pots.append(csasCross)
    rampedParams.append(MultRamp(0, 0, "csasCross.setScale(VALUE)"))

if True:
    from noePotTools import create_NOEPot
    dsts = PotList('dst')
    for (exp, file, scale) in [('noe', "noe_Ail.tbl", 1),
                               ('hbn', "hbn_Ail.tbl", 1)]:
        dst = create_NOEPot(exp,file,)
        dst.setScale(scale)
        dst.setThreshold(0.5)     # dflt [0.5]
        dsts.append(dst)
    pots.append(dsts)
    rampedParams.append(MultRamp(2, 30, "dsts.setScale(VALUE)"))

from xplorPot import XplorPot

cdi_data="cdi_Ail.tbl"
protocol.initDihedrals(cdi_data)
pots.append(XplorPot('CDIH'))
highTempParams.append(StaticRamp("pots['CDIH'].setScale(10)"))
rampedParams.append(StaticRamp("pots['CDIH'].setScale(200)"))

# CONFORMATIONAL ENERGY TERMS.
#===============================================================================
from torsionDBPotTools import create_TorsionDBPot
torsionDB = create_TorsionDBPot(name='torsionDB')
pots.append(torsionDB)
rampedParams.append(MultRamp(0.002, 2, "torsionDB.setScale(VALUE)"))

pots.append(XplorPot('BOND'))          # Dflt scale [1]
pots.append(XplorPot('ANGL'))          # Dflt scale [1]
pots.append(XplorPot('IMPR'))          # Dflt scale [1]
rampedParams.append(MultRamp(0.4, 1,  "pots['BOND'].setScale(VALUE)"))
rampedParams.append(MultRamp(0.4, 1,  "pots['ANGL'].setScale(VALUE)"))
rampedParams.append(MultRamp(0.1, 1,  "pots['IMPR'].setScale(VALUE)"))

# THRESHOLD VALUES FOR  VIOLATION ANALYSIS OF THE POTLIST TERMS.
# Use default values unless specified (cdih [5.0], noe [0.5], rdc [0.5]).
#===============================================================================
pots['BOND'].setThreshold(0.05)     # dflt [0.05]
pots['ANGL'].setThreshold(5.00)     # dflt [2.0]
pots['IMPR'].setThreshold(5.00)     # dflt [2.0]
pots['CDIH'].setThreshold(5.00)     # dflt [5.0]
dsts.setThreshold(5.00)             # dflt [5.0]

# NONBONDED ENERGY TERM - EEFx/IMMx IMPLICIT SOLVATION.
# Membrane hydrophobic thickness: DMPC (25.4A), DPPC (28.6A), POPC (27.0A), DOPC (29.6A)
# See Marsh, Handbook of lipid bilayers, 2nd Ed. p. 379.
# IMMn<10 generates a steeper polar-apolar membrane transition gradient.
# IMMa>0.85 increases dielectric screening (reduces electrostatic effects) in membrane.
# EEFx/IMMx uses nbxmod=4 (exclude 1-2, 1-3 and 1-4 interactions).
#===============================================================================
from eefxPotTools import create_EEFxPot, param_LK
eefx=create_EEFxPot("eefx","All",
                    paramSet=param_LK,
                    verbose=False)
eefx.setScale(1)
eefx.setVerbose(1)
eefx.setIMMx(1)
eefx.useGROUp(1)
eefx.setMoveTol(0.5)
print eefx.showParam()
eefx.setThickness(25) # IMMx membrane thickness [25.4 DMPC; 28.6 DPPC; 27.0 POPC; 29.6 DOPC].
eefx.setProfileN(10)  # IMMx n parameter of membrane profile (use n<3 in early stages).
eefx.setA(0.85)       # IMMx a value that scales dielectric screening. Dflt=[0.85].
pots.append(eefx)

IMM_com = "resid 1:156 and (name CA)"   # Center of mass selection for IMMx position.
Zpos=0									# Z position relative to IMMx membrane center. Dflt=[0].
from eefxPotTools import setCenter, setCenterXY
setCenter(IMM_com, Zpos)				# Translate selected center of mass to IMMx Zpos.


# SET UP IVM OBJECTS (dyn, minc) THAT PERFORM DYNAMICS AND MINIMIZATION.
# IVM (internal variable module) is used to perform dynamics and minimization in
# both torsion-angle and Cartesian space. Bonds, angles and many impropers cannot
# change with internal torsion-angle dynamics.
#===============================================================================
ini_temp = 3000.0 ; fin_temp = 25.0     # Initial and final temperatures.
protocol.massSetup()                    # Give atoms uniform weights except for axes.


from ivm import IVM
dyn = IVM()                             # IVM object for torsion-angle dynamics.
dyn.reset()                             # reset ivm topology for torsion-angle dynamics.
protocol.torsionTopology(dyn)

minc = IVM()                            # IVM object for Cartesian minimization.
protocol.cartesianTopology(minc)

from simulationTools import AnnealIVM
cool = AnnealIVM(initTemp=ini_temp,     # Cooling loop.
                 finalTemp=fin_temp,
                 tempStep=12.5,
                 ivm=dyn,
                 rampedParams=rampedParams)


def accept(potList):
    """
    return True if current structure meets acceptance criteria
    """
    if pots['dsts'].rms()>0.08:
    #if dsts.violations()>0:
        return False
    if pots['CDIH'].rms()>1.75:
    #if pots['CDIH'].violations()>0:
        return False
    if pots['BOND'].violations()>0:
        return False
    if pots['ANGL'].violations()>0:
        return False
    if pots['IMPR'].violations()>1:
        return False
    
    return True


# CALCULATE STRUCTURE MODULE.
#===============================================================================
def calcOneStructure(loopInfo):
    """
    This function calculates a single structure, performs analysis on the
    structure and then writes out a pdb file with remarks.
    """
    InitialParams(tensorInit)       # Calculate initial tensor parameters for this
                                    # structure.
    InitialParams(rampedParams)     # parameters for SA.
    InitialParams(highTempParams)   # reset some rampedParams.

    #===========================================================================
    InitialParams(tensorFinal)
    #setCenter(IMM_com, Zpos)            # translate selected center of mass to IMMx Zpos.
    setCenterXY()           # translate protein coordinates to XY center.
    
    # Do analysis and write structure when this routine is finished.
    pass

# LOOP CONTROL.
#===========================================================================
from simulationTools import StructureLoop, FinalParams
sl=StructureLoop(structLoopAction=calcOneStructure,
              pdbFilesIn=inputStructures,
              numStructures=100,
              pdbTemplate=outFilename,
              calcMissingStructs=True,
              doWriteStructures=True,           # analyze and write coords after calc
              genViolationStats=True,           # print stats file
              averageContext=FinalParams(rampedParams),
              averageFitSel="(%s) and name CA" %# selection for bkbn fit and rmsd [CA]
                              orderedRegion,
                                                # selection for heavy atom rmsd
              averageCompSel="not name H* and (%s)" % orderedRegion, 
              averageTopFraction=0.2,           # Report stats on top 20%
              averagePotList=pots,              # Terms for stats and avg
              averageCrossTerms=[refRMSD],     # Cross validated terms
              averageSortPots=pots,             # Terms used to sort models
              )
sl.run()

import trace
trace.suspend()

if xplor.p_comm.procNum==0:
    
    print "Summary"
    print
    print "precision of structures to mean: %.2f (heavy atom: %.2f)" \
          % (sl.fitRMSD,sl.compRMSD)
    
    refPrec=1.66
    refPrec1=2.7 #all (non-pseudo atom) not known
    ok=True
    if sl.fitRMSD>refPrec:
        print "Error: bad backbone precision"
        ok=False
        pass
    if sl.compRMSD>refPrec1:
        print "Error: bad heavy atom precision"
        ok=False
        pass
    
    bondStats = sl.restraintStats.potTypes['XplorPot']['BOND']
    (bondRMSD,bondRMSDdev) = bondStats.rmsdAveDev()
    
    refBondRMSD=0.01
    if bondRMSD>refBondRMSD:
        print "Error: bad bond rmsd"
        ok=False
        pass
    
    angleStats = sl.restraintStats.potTypes['XplorPot']['ANGL']
    (angleRMSD,angleRMSDdev) = angleStats.rmsdAveDev()
    
    refAngleRMSD=0.5
    if angleRMSD>refAngleRMSD:
        print "Error: bad angle rmsd"
        ok=False
        pass
    
    imprStats = sl.restraintStats.potTypes['XplorPot']['IMPR']
    (imprRMSD,imprRMSDdev) = imprStats.rmsdAveDev()
    
    refImprRMSD=0.4
    if imprRMSD>refImprRMSD:
        print "Error: bad impr rmsd"
        ok=False
        pass
    
    cdihStats = sl.restraintStats.potTypes['XplorPot']['CDIH']
    (cdihRMSD,cdihRMSDdev) = cdihStats.rmsdAveDev()
    
    refCdihRMSD=0.8 # 0.9 quite bad...
    if cdihRMSD>refCdihRMSD:
        print "Error: bad cdih rmsd"
        ok=False
        pass
    
    #FIX: add VDW energy
    #    carbStats = sl.restraintStats.potTypes['XplorPot']['CARB']
    #    (carbRMSD,carbRMSDdev) = carbStats.rmsdAveDev()
    #
    #    refCarbRMSD=1.03
    #    if carbRMSD>refCarbRMSD:
    #        print "Error: bad carb rmsd"
    #        ok=False
    #        pass
    
    noeStats = sl.restraintStats.potTypes['NOEPot']['hbn']
    (noeRMSD,noeRMSDDev) = noeStats.rmsdAveDev()
    
    refNoeRMSD=0.04
    if noeRMSD>refNoeRMSD:
        print "Error: bad hbn rmsd"
        ok=False
        pass
    
    noeStats = sl.restraintStats.potTypes['NOEPot']['noe']
    (noeRMSD,noeRMSDDev) = noeStats.rmsdAveDev()
    
    refNoeRMSD=0.05
    if noeRMSD>refNoeRMSD:
        print "Error: bad noe rmsd"
        ok=False
        pass
    
    accStats = sl.restraintStatsCross.potTypes['PosDiffPot']['refRMSD']
    (accRMSD,accRMSDDev) = accStats.rmsdAveDev()
    
    refAcc=1.3
    if accRMSD>refAcc:
        print "Error: bad accuracy to reference: %.3f" % accRMSD
        ok=False
        pass

    if not ok:
        raise Exception("low quality results")

    open("ok.tmp","w").write("validation succeeded")


