
import protocol

opts,args = xplor.parseArguments(["quick:0"])
quick=True if opts and opts[0][0] == "quick" else False

# Total number of structures - 100 structures from 10 starting structures
numberOfStructures=3 if quick else 100

inTemplate="fold" + "_*.sa.best"
# Base name for output pdb files.
outfilename = 'SCRIPT_STRUCTURE.sa'

# Initialize random seed.
protocol.initRandomSeed(3421)

import eefxPotTools
eefxPotTools.initEEFx()

# load PDB to generate PSF
import glob
protocol.loadPDB( glob.glob(inTemplate)[0],
                  deleteUnknownAtoms=True )
                  

#list to hold potential terms in structure calculation
from potList import PotList
potList = PotList()
crossTerms = PotList('cross')

#compute RMSD to reference structure
#  backbone and heavy atom RMSDs will be printed in the output
#  structure files
#
from posDiffPotTools import create_PosDiffPot
refRMSD = create_PosDiffPot("refRMSD",
                            "(name CA or name C or name N) and resid 1:230",
                            pdbFile='1EZA.pdb',
                            cmpSel="not name H*")


# lists to hold force constant settings for various stages of
# structure calculation
from simulationTools import MultRamp, StaticRamp
hiTempParams = []
# Settings for annealing stage.
rampedParams = []

# Set up NOE potential.
from noePotTools import create_NOEPot
noe = create_NOEPot('noe', 'noe.tbl')
potList.append(noe)
rampedParams.append(MultRamp(2.0,30.0, "noe.setScale( VALUE )"))

# Set up dihedral angles.
torsionFile='dihedral.tbl'
protocol.initDihedrals(torsionFile)
from xplorPot import XplorPot
potList.append(XplorPot('CDIH'))
hiTempParams.append( StaticRamp("potList['CDIH'].setScale(10)") )
rampedParams.append( StaticRamp("potList['CDIH'].setScale(200)") )

# Statistical torsion angle potential
#
from torsionDBPotTools import create_TorsionDBPot
torsionDBPot = create_TorsionDBPot('tDB')
potList.append( torsionDBPot )
rampedParams.append( MultRamp(.002,2,"torsionDBPot.setScale(VALUE)") )

#covalent terms
from xplorPot import XplorPot
for term in ('BOND', 'ANGL', 'IMPR'):
    potList.append( XplorPot(term) )
    pass

# Set threshold for terms in potList to allow violation analysis.
potList['ANGL'].setThreshold(5.0)   # default is 2.0
potList['IMPR'].setThreshold(5.0)   # default is 2.0
# Use default values for the rest (bond: 0.05, cdih: 5.0, noe: 0.5).

rampedParams.append(MultRamp(0.4, 1.0, "potList['ANGL'].setScale(VALUE)"))
rampedParams.append(MultRamp(0.1, 1.0, "potList['IMPR'].setScale(VALUE)"))

#nonbonded term
from eefxPotTools import create_EEFxPot, param_LK
eefxpot=create_EEFxPot("eefxpot")
eefxpot.setVerbose(False)
potList.append(eefxpot)
rampedParams.append(MultRamp(0.1,1.0,"eefxpot.setScale(VALUE)"))

##use repel at high temp
#rampedParams.append(StaticRamp("potList['VDW'].setScale(0)"))
#potList.append( XplorPot("VDW") )
#hiTempParams.append(StaticRamp("""protocol.initNBond(cutnb=100,
#                                                     rcon=0.04,
#                                                     tolerance=0.5,
#                                                     nbxmod=5,
#                                                     repel=0.9,)"""))  
#hiTempParams.append(StaticRamp("xplor.command('param nbonds rcon 1 end end')"))
#rampedParams.append(StaticRamp("potList['VDW'].setScale(1)"))
#hiTempParams.append(StaticRamp("eefxpot.setScale(0)"))     


# Set up IVM object(s).

# IVM object for torsion-angle dynamics.
from ivm import IVM
dyn = IVM()
protocol.torsionTopology(dyn)

# IVM object for final Cartesian minimization.
minc = IVM()
protocol.cartesianTopology(minc)



temp_ini = 3500.0
temp_fin = 25.0


# Give atoms uniform weights, except for the anisotropy axis.
protocol.massSetup()


def calcOneStructure(loopInfo):
    """Calculate a structure.

    """
    # initial structure specified by inTemplate.
#    protocol.fixupCovalentGeom(maxIters=100, useVDW=1)


    # Initialize parameters for high temp dynamics.
    from simulationTools import InitialParams
    InitialParams(rampedParams) # overides some in rampedParams
    InitialParams(hiTempParams) # overides some in rampedParams

    #
    # High Temperature Dynamics Stage.
    #


    protocol.initDynamics(dyn,
                          potList=potList,
                          bathTemp=temp_ini,
                          initVelocities=True,
                          finalTime=15,  # run for finalTime or 
                                         # numSteps * 0.001, whichever is less
                          numSteps=50 if quick else 15001,
                          printInterval=100)


    dyn.setETolerance(temp_ini/100) # used to det. stepsize. default: temp/1000
    dyn.run()

                          
    #
    # Simulated Annealing Stage.
    #
    # Initialize parameters for annealing.
    InitialParams(rampedParams)

    # Set up IVM object for annealing.
    protocol.initDynamics(dyn,
                          potList=potList,
                          finalTime=0.4,  # run for finalTime or 
                                          # numSteps * 0.001, whichever is less
                          numSteps=2 if quick else 201,   
                          printInterval=100)    

    # Set up cooling loop and run.
    from simulationTools import AnnealIVM
    AnnealIVM(initTemp=temp_ini,
              finalTemp=temp_fin,
              tempStep=1000 if quick else 12.5,
              ivm=dyn,
              rampedParams=rampedParams).run()

    #
    # Torsion angle minimization.
    #
    protocol.initMinimize(dyn,
                          potList=potList,
                          printInterval=50)
    if not quick: dyn.run()

    #
    # Final Cartesian minimization.
    #
    protocol.initMinimize(minc,
                          potList=potList,
                          dEPred=10)
    if not quick: minc.run()

    # Write structure.
    return



from simulationTools import FinalParams, StructureLoop
StructureLoop(numStructures=numberOfStructures,
              pdbFilesIn=inTemplate,
              doWriteStructures=True,
              averagePotList=potList,
              averageContext=FinalParams(rampedParams),
              pdbTemplate=outfilename,
              structLoopAction=calcOneStructure,
              genViolationStats=True,
              averageCrossTerms=refRMSD,
              averageTopFraction=0.20, #report only on best 20% of structs
              averageFilename="SCRIPT_ave.pdb",   #generate regularized ave structure
              ).run()
                              
